<?php

namespace App\Extensions\MarketingBot\System\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\User;

class ShopifyStore extends Model
{
    protected $table = 'ext_shopify_stores';

    protected $fillable = [
        'user_id',
        'shop_domain',
        'access_token',
        'api_key',
        'api_secret',
        'is_active',
        'last_sync_at',
        'last_product_sync_at',
        'last_order_sync_at',
        'sync_settings',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'last_sync_at' => 'datetime',
        'last_product_sync_at' => 'datetime',
        'last_order_sync_at' => 'datetime',
        'sync_settings' => 'array',
    ];

    protected $hidden = [
        'access_token',
        'api_secret',
    ];

    /**
     * Get the user that owns the Shopify store
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get all products for this store
     */
    public function products(): HasMany
    {
        return $this->hasMany(ShopifyProduct::class);
    }

    /**
     * Get all orders for this store
     */
    public function orders(): HasMany
    {
        return $this->hasMany(ShopifyOrder::class);
    }

    /**
     * Get active products
     */
    public function activeProducts(): HasMany
    {
        return $this->products()->where('is_active', true);
    }

    /**
     * Check if store needs sync
     */
    public function needsSync(string $type = 'general', int $minutes = 60): bool
    {
        $lastSyncField = match($type) {
            'products' => 'last_product_sync_at',
            'orders' => 'last_order_sync_at',
            default => 'last_sync_at',
        };

        if (!$this->$lastSyncField) {
            return true;
        }

        return $this->$lastSyncField->diffInMinutes(now()) >= $minutes;
    }

    /**
     * Mark sync as completed
     */
    public function markSynced(string $type = 'general'): void
    {
        $field = match($type) {
            'products' => 'last_product_sync_at',
            'orders' => 'last_order_sync_at',
            default => 'last_sync_at',
        };

        $this->update([$field => now()]);
    }
}
